/** @file   minigunalien.cpp
 * @brief   Implementation of MinigunAlien - class.
 * @version $Revision: 1.4 $
 * @author  Tomi Lamminsaari
 */

#include "minigunalien.h" // class's header file
#include "minigunai.h"
#include "gameanims.h"
#include "warglobals.h"
#include "animplayer.h"
#include "GfxManager.h"
#include "www_map.h"
#include "settings.h"
#include "soundsamples.h"
#include "AnimId.h"
using namespace eng2d;


namespace WeWantWar {


const int MinigunAlien::GUN_CONTROLPOINT_INDEX = 0;


/** Constructor.
 */
MinigunAlien::MinigunAlien() :
  Alien( ),
  
  m_speedVec( 0, 0 )
{
  ObjectID::Type oid = ObjectID::TYPE_MINIGUNALIEN;
  
  // Set properties
  m_speedVec.y( - Settings::floatObjProp(oid, "spd_walk:") );
  this->setCorrectAnimation( GameAnims::EIdle );
	
  this->boundingSphere( Settings::floatObjProp(oid, "bounding_sphere:") );
  this->setArmor( Settings::floatObjProp(oid, "armor:") );
  this->setCollisionPoint( 0, Vec2D( -13,-13 ) );
  this->setCollisionPoint( 1, Vec2D(  13,-13 ) );
  this->setCollisionPoint( 2, Vec2D(  13, 13 ) );
  this->setCollisionPoint( 3, Vec2D( -13, 13 ) );
  
  // Add the controlpoint for the minigun head
  this->addCtrlPoint( Vec2D( 2, -18 ) );
  
  // Create the ai object.
  this->setController( new MinigunAI( this ) );
}

/** Destructor.
 */
MinigunAlien::~MinigunAlien()
{
}



/** Updates this alien
 */
void MinigunAlien::update()
{
  if ( this->state() == GameObject::STATE_KILLED ) {
    return;
  }
  
  if ( this->state() == GameObject::STATE_DYING ) {
    // We're dying, so update the dying animation
    if ( m_animation.paused() ) {
      // The dying animation has been run through, so set this object
      // to killed-state. This alien drops the Minigun.
      this->state( GameObject::STATE_KILLED );
      this->setCorrectAnimation( GameAnims::EKilled );

      BonusObject* pBonus = new BonusObject( BonusObject::MINIGUN );
      pBonus->position( this->position() );
      pBonus->size( 22 );
      WarGlobals::bonusList.push_back( pBonus );
    }
    return;

  }

  // This alien is alive.
  m_pController->update();

  if ( m_pController->forward() == 0 ) {
    this->setCorrectAnimation( GameAnims::EIdle );
  }

  this->changeAngle( m_pController->turn() );
  if ( m_pController->forward() == 1 ) {
    Vec2D spdV( m_speedVec );
    spdV.rotate( this->angle() );
    this->move( spdV );
  }

  if ( m_pController->shoot() == 1 ) {
    this->attack();

  } else if ( this->getAnimID() != GameAnims::EWalk &&
              m_pController->forward() != 0 ) {
      this->setCorrectAnimation( GameAnims::EWalk );
  }
}




/** Kills this alien
 */
void MinigunAlien::kill()
{
  this->state( GameObject::STATE_DYING );
  this->setCorrectAnimation( GameAnims::EDying );
  this->makeSound( GameObject::SND_DIE );
}



/** Makes sound
 */
void MinigunAlien::makeSound( GameObject::SoundID id ) const
{
  if ( id == GameObject::SND_ATTACK ) {
    Sound::playSample( SMP_WEAPON4, false );
  }
  if ( id == GameObject::SND_PAIN ) {
    int t = ( rand() % 2 < 1 ) ? SMP_ALIEN1_PAIN : SMP_ALIEN1_PAIN2;
    Sound::playSample( t, false );
  }
  if ( id == GameObject::SND_DIE ) {
    Sound::playSample( SMP_ALIEN1_DIE, false );
  }
}



/** Hit by bullet
 */
bool MinigunAlien::hitByBullet( Bullet* pB )
{
  bool ret = Alien::hitByBullet( pB );
  if ( ret == true ) {
    if ( pB->iType != Bullet::EFlameThrower ) {
      ParticleBlood* pP = new ParticleBlood( pB->iPosition, pB->velocity(), 12,
                                             Color(130,60,10) );
      WarGlobals::pPartManager->addSystem( pP );
    }
  }
  
  // Inform the AI-controller that we got hit.
  AIController* pC = dynamic_cast<AIController*>( this->getController() );
  AIController::BFlag f = pC->getFlags();
  f &= ~AIController::WAIT_TARGET;
  pC->setFlags( f );
  return ret;
}



///
/// Getter methods
/// ==============

/** Returns the type of this alien.
 */
ObjectID::Type MinigunAlien::objectType() const
{
  return ObjectID::TYPE_MINIGUNALIEN;
}



///
/// Private and proteceted methods
/// ==============================

/** Sets the requested animation
 */
void MinigunAlien::setCorrectAnimation( int aAnimId )
{
  if ( this->getAnimID() != aAnimId ) {
    const Animation& anim = GameAnims::findAnimation( AnimId::KMinigunAlien, aAnimId );
    this->setAnimation( anim, aAnimId );
  }
}



/** Shoots
 */
void MinigunAlien::attack()
{
  if ( this->reloading() == false ) {
    // Handle shooting

    Vec2D gunV = this->getCtrlPoint( GUN_CONTROLPOINT_INDEX );
    gunV += this->position();

    Weapon::Specs& w = Weapon::getWeaponSpecs( Weapon::W_MINIGUN );
    Bullet* pB = BulletTable::createBullet( this, gunV, Bullet::EMinigun );
    pB->iDamage /= 2;
    WarGlobals::pBulletManager->spawnBullet( pB );

    ParticleSystem* particles = new ParticleGunFlames( gunV, pB->velocity(), 15, 60 );
    WarGlobals::pPartManager->addSystem( particles );
    
    const Animation& rifleFlameAnim =
        GameAnims::findAnimation(AnimId::KMinigunShootFlame, 0);
    AnimPlayer::spawn( rifleFlameAnim, gunV, 0 );
    const Animation& lightAnim = GameAnims::findAnimation( AnimId::KRifleShotLight );
    if ( Settings::shootingLights == true ) {
      AnimPlayer::spawn( lightAnim, gunV, 0 );
    }
    this->setCounter( RELOAD_COUNTER_INDEX, w.reload );
    this->makeSound( GameObject::SND_ATTACK );
  }
  this->setCorrectAnimation( GameAnims::EShoot );
}


} // end of namespace
